<?php
require_once __DIR__ . '/util.php';

function prisma_token_from_db(): string {
  // Optional: token stored in DB settings (admin UI)
  try {
    require_once __DIR__ . '/db.php';
    $st = $db->prepare("SELECT v FROM settings WHERE k='PRISMA_BEARER_TOKEN_DB'");
    $st->execute();
    $v = (string)($st->fetchColumn() ?: '');
    return trim($v);
  } catch (Exception $e) {
    return '';
  }
}


function prisma_get_bearer(): string {
  // 1) Token fijo (config)
  if (PRISMA_BEARER_TOKEN) return PRISMA_BEARER_TOKEN;

  // 1.b) Token guardado en DB (admin)
  $dbTok = prisma_token_from_db();
  if ($dbTok) return $dbTok;

  // 2) OAuth2 client_credentials (si está configurado)
  if (!PRISMA_TOKEN_URL || !PRISMA_CLIENT_ID || !PRISMA_CLIENT_SECRET) {
    throw new Exception("Prisma token no configurado. Setear PRISMA_BEARER_TOKEN o PRISMA_TOKEN_URL + CLIENT_ID/SECRET en config.php.");
  }

  // Cache simple en session (para hosting)
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  $cached = $_SESSION['_prisma_token'] ?? null;
  if ($cached && ($cached['exp'] ?? 0) > time()+30) {
    return $cached['token'];
  }

  $post = [
    'grant_type' => 'client_credentials',
    'client_id' => PRISMA_CLIENT_ID,
    'client_secret' => PRISMA_CLIENT_SECRET,
  ];
  if (PRISMA_TOKEN_AUDIENCE) $post['audience'] = PRISMA_TOKEN_AUDIENCE;
  if (PRISMA_TOKEN_SCOPE) $post['scope'] = PRISMA_TOKEN_SCOPE;

  $ch = curl_init(PRISMA_TOKEN_URL);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => ["Accept: application/json"],
    CURLOPT_POSTFIELDS => http_build_query($post),
    CURLOPT_TIMEOUT => 30,
  ]);
  $raw = curl_exec($ch);
  $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  $data = json_decode($raw, true);
  if ($http >= 400 || !is_array($data)) {
    throw new Exception("Error obteniendo token Prisma (HTTP $http).");
  }
  $token = $data['access_token'] ?? '';
  $expiresIn = (int)($data['expires_in'] ?? 300);
  if (!$token) throw new Exception("Token Prisma inválido.");

  $_SESSION['_prisma_token'] = ['token'=>$token,'exp'=>time()+$expiresIn];
  return $token;
}

function prisma_request(string $method, string $url, array $body=null): array {
  $token = prisma_get_bearer();
  $ch = curl_init($url);
  $headers = [
    "Authorization: Bearer {$token}",
    "Content-Type: application/json",
    "Accept: application/json",
  ];
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_CUSTOMREQUEST => $method,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_TIMEOUT => 30,
  ]);
  if ($body !== null) {
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body, JSON_UNESCAPED_UNICODE));
  }
  $raw = curl_exec($ch);
  $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  $data = json_decode($raw, true);
  if ($http >= 400) {
    $msg = is_array($data) ? ($data['mensaje'] ?? $data['message'] ?? 'Error') : 'Error';
    throw new Exception("Prisma HTTP {$http}: {$msg}");
  }
  return is_array($data) ? $data : [];
}

function prisma_debts_details(array $payload): array {
  return prisma_request('POST', PRISMA_CORE_BASE . '/debts_details', $payload);
}

function prisma_agents_payments(array $payload): array {
  return prisma_request('POST', PRISMA_AGENT_BASE . '/agents-payments', $payload);
}

function prisma_payment_status(array $payload): array {
  return prisma_request('POST', PRISMA_CORE_BASE . '/payment_status', $payload);
}
